<?php

declare(strict_types=1);

namespace App\Bot;

final class TelegramClient
{
    private string $token;
    private string $apiBase;

    public function __construct(string $token)
    {
        $this->token = $token;
        $this->apiBase = 'https://api.telegram.org/bot' . $token . '/';
    }

    public function sendMessage(int $chatId, string $text, array $params = []): array
    {
        $payload = array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ], $params);

        return $this->request('sendMessage', $payload);
    }

    public function editMessageText(int $chatId, int $messageId, string $text, array $params = []): void
    {
        $params['chat_id'] = $chatId;
        $params['message_id'] = $messageId;
        $params['text'] = $text;

        $this->request('editMessageText', $params);
    }

    public function deleteMessage(int $chatId, int $messageId): void
    {
        $this->request('deleteMessage', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
        ]);
    }

    public function request(string $method, array $params = []): array
    {
        $url = $this->apiBase . $method;

        if (isset($params['reply_markup']) && is_array($params['reply_markup'])) {
            $encoded = json_encode($params['reply_markup'], JSON_UNESCAPED_UNICODE);
            if ($encoded !== false) {
                $params['reply_markup'] = $encoded;
            } else {
                unset($params['reply_markup']);
            }
        }

        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => $params,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => 0,
            ]);
            $response = curl_exec($ch);
            if ($response === false) {
                $error = curl_error($ch);
                throw new \RuntimeException('Telegram request failed: ' . $error);
            }
        } else {
            $context = stream_context_create([
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-type: application/x-www-form-urlencoded',
                    'content' => http_build_query($params),
                    'timeout' => 30,
                ],
            ]);
            $response = file_get_contents($url, false, $context);
            if ($response === false) {
                throw new \RuntimeException('Telegram request failed.');
            }
        }

        $data = json_decode($response, true);
        if (!is_array($data)) {
            throw new \RuntimeException('Invalid Telegram response.');
        }

        if (array_key_exists('ok', $data) && $data['ok'] === false) {
            $description = isset($data['description']) ? (string) $data['description'] : 'Unknown Telegram API error.';
            throw new \RuntimeException('Telegram API error: ' . $description);
        }

        return $data;
    }
}
