<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class AdminModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return $command === '/admin'
            || $command === '/panel'
            || $command === '/install_group'
            || $command === '/set'
            || $command === '/student';
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
        if ($command === '/install_group') {
            $this->handleInstallGroup($update, $bot);
            return;
        }

        if ($command === '/set') {
            $this->handleSetTopicRole($update, $bot);
            return;
        }

        if ($command === '/student') {
            $this->handleStudentLookupCommand($update, $bot);
            return;
        }

        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];

        $text = "<b>پنل ادمین ربات تریدینگ</b>\n";
        $text .= "از دکمه‌های زیر برای مشاهده داشبورد، مدیریت گروه اصلی و آمار هنرجویان استفاده کن.";

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '📊 داشبورد ادمین', 'callback_data' => 'admin:dashboard'],
                    ['text' => '👥 مدیریت گروه‌ها', 'callback_data' => 'admin:groups'],
                ],
                [
                    ['text' => '👤 هنرجویان', 'callback_data' => 'admin:students'],
                    ['text' => '🚨 هنرجویان غیرفعال', 'callback_data' => 'inactivity:summary'],
                ],
                [
                    ['text' => '🧠 تحلیل رفتاری', 'callback_data' => 'behavior:summary'],
                    ['text' => '🎯 اولویت توجه کوچ', 'callback_data' => 'coach:summary'],
                ],
                [
                    ['text' => '🏆 لیدربورد برترین‌ها', 'callback_data' => 'leaderboard:summary'],
                ],
            ],
        ];

        $bot->getClient()->sendMessage($chatId, $text, [
            'reply_markup' => $keyboard,
        ]);
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'student:') === 0) {
            $this->handleStudentStatsCallback($data, $update, $bot);
            return;
        }

        if (strpos($data, 'admin:') !== 0) {
            return;
        }

        $action = substr($data, strlen('admin:'));

        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];
        $messageId = isset($message['message_id']) ? (int) $message['message_id'] : 0;
        $storage = $bot->getStorage();

        if ($action === 'menu') {
            $text = "<b>پنل ادمین ربات تریدینگ</b>\n";
            $text .= "از دکمه‌های زیر برای مشاهده داشبورد، مدیریت گروه اصلی و آمار هنرجویان استفاده کن.";

            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '📊 داشبورد ادمین', 'callback_data' => 'admin:dashboard'],
                        ['text' => '👥 مدیریت گروه‌ها', 'callback_data' => 'admin:groups'],
                    ],
                    [
                        ['text' => '👤 هنرجویان', 'callback_data' => 'admin:students'],
                        ['text' => '🚨 هنرجویان غیرفعال', 'callback_data' => 'inactivity:summary'],
                    ],
                    [
                        ['text' => '🧠 تحلیل رفتاری', 'callback_data' => 'behavior:summary'],
                        ['text' => '🎯 اولویت توجه کوچ', 'callback_data' => 'coach:summary'],
                    ],
                    [
                        ['text' => '🏆 لیدربورد برترین‌ها', 'callback_data' => 'leaderboard:summary'],
                    ],
                ],
            ];

            if ($messageId > 0) {
                $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            } else {
                $bot->getClient()->sendMessage($chatId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            }

            return;
        }

        if ($action === 'dashboard') {
            $users = $storage->load('users');
            $topicActivity = $storage->load('topic_activity');

            $totalUsers = is_array($users) ? count($users) : 0;

            $groupsCount = 0;
            $topicsCount = 0;
            $messagesTotal = 0;
            $perUser = [];

            $adminIdList = array_map('intval', $adminIds);

            if (is_array($topicActivity)) {
                foreach ($topicActivity as $chatTopics) {
                    if (!is_array($chatTopics)) {
                        continue;
                    }
                    $groupsCount++;
                    $topicsCount += count($chatTopics);
                    foreach ($chatTopics as $topic) {
                        if (isset($topic['messages_total'])) {
                            $messagesTotal += (int) $topic['messages_total'];
                        }

                        if (!is_array($topic) || !isset($topic['users']) || !is_array($topic['users'])) {
                            continue;
                        }

                        foreach ($topic['users'] as $uid => $u) {
                            $uid = (int) $uid;
                            if (in_array($uid, $adminIdList, true)) {
                                continue;
                            }
                            if (!isset($perUser[$uid])) {
                                $perUser[$uid] = [
                                    'total_messages' => 0,
                                    'last_message_at' => '',
                                ];
                            }

                            $count = (int) ($u['message_count'] ?? 0);
                            $perUser[$uid]['total_messages'] += $count;

                            $uLast = (string) ($u['last_message_at'] ?? '');
                            if ($uLast !== '' && ($perUser[$uid]['last_message_at'] === '' || $uLast > $perUser[$uid]['last_message_at'])) {
                                $perUser[$uid]['last_message_at'] = $uLast;
                            }
                        }
                    }
                }
            }

            $settings = $storage->load('settings');
            $mainGroupText = '';
            if (isset($settings['main_group']['id'])) {
                $mg = $settings['main_group'];
                $mainGroupText .= "<b>گروه اصلی ثبت‌شده:</b>\n";
                $mainGroupText .= 'ID: <code>' . (int) ($mg['id'] ?? 0) . '</code>' . "\n";
                $mainGroupText .= 'عنوان: ' . (string) ($mg['title'] ?? '-') . "\n";
            } else {
                $mainGroupText .= "هنوز گروه اصلی ثبت نشده است.\n";
                $mainGroupText .= "برای ثبت، ربات را به گروه ترید اضافه کن و در آن گروه بنویس: <b>نصب ربات</b>.";
            }

            $activeCount = 0;
            $semiCount = 0;
            $inactiveCount = 0;

            if (!empty($perUser)) {
                $now = new \DateTimeImmutable('now');
                foreach ($perUser as $u) {
                    $lastAt = $u['last_message_at'];
                    $days = null;
                    if ($lastAt !== '') {
                        try {
                            $dt = new \DateTimeImmutable($lastAt);
                            $diff = $now->diff($dt);
                            $days = (int) $diff->days;
                        } catch (\Throwable $e) {
                            $days = null;
                        }
                    }

                    $total = (int) $u['total_messages'];
                    if ($days !== null && $days <= 7 && $total >= 20) {
                        $activeCount++;
                    } elseif ($days !== null && $days <= 30 && $total >= 5) {
                        $semiCount++;
                    } else {
                        $inactiveCount++;
                    }
                }
            }

            $text = "<b>📊 داشبورد کلی ربات</b>\n\n";
            $text .= '👥 <b>کاربران ثبت‌شده:</b> ' . $totalUsers . "\n";
            $text .= '💬 <b>گروه‌های دارای تاپیک:</b> ' . $groupsCount . "\n";
            $text .= '🧵 <b>تعداد تاپیک‌ها:</b> ' . $topicsCount . "\n";
            $text .= '✉️ <b>مجموع پیام‌های ثبت‌شده در تاپیک‌ها:</b> ' . $messagesTotal . "\n";
            $text .= '🔥 <b>هنرجویان فعال:</b> ' . $activeCount . ' | 🟡 نیمه‌فعال: ' . $semiCount . ' | ⚫ غیرفعال: ' . $inactiveCount . "\n\n";
            $text .= $mainGroupText . "\n\n";
            $text .= "برای مشاهده جزئیات هنرجویان از دکمه <b>👤 هنرجویان</b> استفاده کن.";

            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '⬅️ بازگشت به منوی ادمین', 'callback_data' => 'admin:menu'],
                    ],
                ],
            ];

            if ($messageId > 0) {
                $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            } else {
                $bot->getClient()->sendMessage($chatId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            }
        } elseif ($action === 'groups') {
            $settings = $storage->load('settings');

            if (isset($settings['main_group']['id'])) {
                $mg = $settings['main_group'];
                $text = "گروه اصلی فعلی:\n";
                $text .= 'ID: ' . (int) ($mg['id'] ?? 0) . "\n";
                $text .= 'عنوان: ' . (string) ($mg['title'] ?? '-') . "\n\n";
                $text .= "برای تغییر، ربات را به گروه جدید اضافه کن و در آن گروه بنویس: \"نصب ربات\".\n";
                $text .= "برای ریست کامل گروه اصلی و آمار آن، از دکمه زیر استفاده کن.";

                $keyboard = [
                    'inline_keyboard' => [
                        [
                            ['text' => '🔁 ریست گروه اصلی', 'callback_data' => 'admin:groups_reset'],
                        ],
                        [
                            ['text' => '⬅️ بازگشت به منوی ادمین', 'callback_data' => 'admin:menu'],
                        ],
                    ],
                ];

                if ($messageId > 0) {
                    $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                        'reply_markup' => $keyboard,
                    ]);
                } else {
                    $bot->getClient()->sendMessage($chatId, $text, [
                        'reply_markup' => $keyboard,
                    ]);
                }
                return;
            } else {
                $text = "هنوز هیچ گروهی به عنوان گروه اصلی ثبت نشده است.\n";
                $text .= "ربات را به گروه ترید اضافه کن و در آن گروه بنویس: \"نصب ربات\" تا به عنوان گروه اصلی ذخیره شود.";
            }

            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '⬅️ بازگشت به منوی ادمین', 'callback_data' => 'admin:menu'],
                    ],
                ],
            ];

            if ($messageId > 0) {
                $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            } else {
                $bot->getClient()->sendMessage($chatId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            }
        } elseif ($action === 'groups_reset') {
            $settings = $storage->load('settings');
            if (!isset($settings['main_group']['id'])) {
                $bot->getClient()->sendMessage($chatId, 'هیچ گروه اصلی برای ریست وجود ندارد.');
                return;
            }

            $mainGroupId = (int) $settings['main_group']['id'];

            $storage->update('settings', function (array $s) use ($mainGroupId): array {
                unset($s['main_group']);
                return $s;
            });

            $storage->update('topic_roles', function (array $roles) use ($mainGroupId): array {
                unset($roles[$mainGroupId]);
                return $roles;
            });

            $storage->update('topic_activity', function (array $activity) use ($mainGroupId): array {
                unset($activity[$mainGroupId]);
                return $activity;
            });

            $storage->update('ratings', function (array $ratings) use ($mainGroupId): array {
                unset($ratings[$mainGroupId]);
                return $ratings;
            });

            $bot->getClient()->sendMessage($chatId, 'گروه اصلی و آمار مربوط به آن ریست شد. برای شروع دوباره، ربات را در گروه جدید نصب کن.');
        } elseif ($action === 'students') {
            $settings = $storage->load('settings');
            if (!isset($settings['main_group']['id'])) {
                $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
                return;
            }

            $mainGroupId = (int) $settings['main_group']['id'];

            $topicActivity = $storage->load('topic_activity');
            $topicRoles = $storage->load('topic_roles');
            $ratings = $storage->load('ratings');
            $users = $storage->load('users');

            $adminIdList = array_map('intval', $adminIds);

            $perUser = [];

            $rolesForChat = $topicRoles[$mainGroupId] ?? [];
            $tradingTopicId = isset($rolesForChat['trading']['topic_id']) ? (int) $rolesForChat['trading']['topic_id'] : null;
            $warTopicId = isset($rolesForChat['war']['topic_id']) ? (int) $rolesForChat['war']['topic_id'] : null;
            $qaTopicId = isset($rolesForChat['qa']['topic_id']) ? (int) $rolesForChat['qa']['topic_id'] : null;

            if (isset($topicActivity[$mainGroupId]) && is_array($topicActivity[$mainGroupId])) {
                foreach ($topicActivity[$mainGroupId] as $topicId => $topic) {
                    if (!is_array($topic) || !isset($topic['users']) || !is_array($topic['users'])) {
                        continue;
                    }

                    $role = 'other';
                    if ($tradingTopicId !== null && (int) $topicId === $tradingTopicId) {
                        $role = 'trading';
                    } elseif ($warTopicId !== null && (int) $topicId === $warTopicId) {
                        $role = 'war';
                    } elseif ($qaTopicId !== null && (int) $topicId === $qaTopicId) {
                        $role = 'qa';
                    }

                    foreach ($topic['users'] as $uid => $u) {
                        $uid = (int) $uid;
                        if (in_array($uid, $adminIdList, true)) {
                            continue;
                        }
                        if (!isset($perUser[$uid])) {
                            $perUser[$uid] = [
                                'id' => $uid,
                                'first_name' => (string) ($u['first_name'] ?? ''),
                                'last_name' => (string) ($u['last_name'] ?? ''),
                                'username' => (string) ($u['username'] ?? ''),
                                'total_messages' => 0,
                                'messages_trading' => 0,
                                'messages_war' => 0,
                                'messages_qa' => 0,
                                'messages_other' => 0,
                                'first_message_at' => (string) ($u['first_message_at'] ?? ''),
                                'last_message_at' => (string) ($u['last_message_at'] ?? ''),
                                'rated_analyses' => 0,
                                'ratings_sum' => 0,
                                'ratings_count' => 0,
                                'votes_cast' => 0,
                                'last_vote_at' => '',
                            ];
                        }

                        $count = (int) ($u['message_count'] ?? 0);
                        $perUser[$uid]['total_messages'] += $count;
                        if ($role === 'trading') {
                            $perUser[$uid]['messages_trading'] += $count;
                        } elseif ($role === 'war') {
                            $perUser[$uid]['messages_war'] += $count;
                        } elseif ($role === 'qa') {
                            $perUser[$uid]['messages_qa'] += $count;
                        } else {
                            $perUser[$uid]['messages_other'] += $count;
                        }

                        $uFirst = (string) ($u['first_message_at'] ?? '');
                        $uLast = (string) ($u['last_message_at'] ?? '');

                        if ($perUser[$uid]['first_message_at'] === '' || ($uFirst !== '' && $uFirst < $perUser[$uid]['first_message_at'])) {
                            $perUser[$uid]['first_message_at'] = $uFirst;
                        }
                        if ($uLast !== '' && ($perUser[$uid]['last_message_at'] === '' || $uLast > $perUser[$uid]['last_message_at'])) {
                            $perUser[$uid]['last_message_at'] = $uLast;
                        }
                    }
                }
            }

            if (isset($ratings[$mainGroupId]) && is_array($ratings[$mainGroupId])) {
                foreach ($ratings[$mainGroupId] as $entry) {
                    if (!is_array($entry)) {
                        continue;
                    }
                    $authorId = isset($entry['author_id']) ? (int) $entry['author_id'] : 0;
                    if ($authorId !== 0 && !in_array($authorId, $adminIdList, true)) {
                        if (!isset($perUser[$authorId])) {
                            $perUser[$authorId] = [
                                'id' => $authorId,
                                'first_name' => '',
                                'last_name' => '',
                                'username' => '',
                                'total_messages' => 0,
                                'messages_trading' => 0,
                                'messages_war' => 0,
                                'messages_qa' => 0,
                                'messages_other' => 0,
                                'first_message_at' => '',
                                'last_message_at' => '',
                                'rated_analyses' => 0,
                                'ratings_sum' => 0,
                                'ratings_count' => 0,
                                'votes_cast' => 0,
                                'last_vote_at' => '',
                            ];
                        }

                        $perUser[$authorId]['rated_analyses']++;

                        $statsEntry = $entry['stats'] ?? [];
                        $sum = (int) ($statsEntry['sum'] ?? 0);
                        $count = (int) ($statsEntry['count'] ?? 0);

                        $perUser[$authorId]['ratings_sum'] += $sum;
                        $perUser[$authorId]['ratings_count'] += $count;
                    }

                    if (isset($entry['votes']) && is_array($entry['votes'])) {
                        foreach ($entry['votes'] as $voterId => $vote) {
                            $voterId = (int) $voterId;
                            if (in_array($voterId, $adminIdList, true)) {
                                continue;
                            }
                            if (!isset($perUser[$voterId])) {
                                $perUser[$voterId] = [
                                    'id' => $voterId,
                                    'first_name' => '',
                                    'last_name' => '',
                                    'username' => '',
                                    'total_messages' => 0,
                                    'messages_trading' => 0,
                                    'messages_war' => 0,
                                    'messages_qa' => 0,
                                    'messages_other' => 0,
                                    'first_message_at' => '',
                                    'last_message_at' => '',
                                    'rated_analyses' => 0,
                                    'ratings_sum' => 0,
                                    'ratings_count' => 0,
                                    'votes_cast' => 0,
                                    'last_vote_at' => '',
                                ];
                            }

                            if (!isset($perUser[$voterId]['votes_cast'])) {
                                $perUser[$voterId]['votes_cast'] = 0;
                            }
                            $perUser[$voterId]['votes_cast']++;

                            $votedAt = (string) ($vote['voted_at'] ?? '');
                            if ($votedAt !== '') {
                                if (!isset($perUser[$voterId]['last_vote_at']) || $perUser[$voterId]['last_vote_at'] === '' || $votedAt > $perUser[$voterId]['last_vote_at']) {
                                    $perUser[$voterId]['last_vote_at'] = $votedAt;
                                }
                            }
                        }
                    }
                }
            }

            if (empty($perUser)) {
                $bot->getClient()->sendMessage($chatId, 'هنوز هیچ فعالیتی برای هنرجویان در گروه اصلی ثبت نشده است.');
                return;
            }

            $now = new \DateTimeImmutable('now');

            foreach ($perUser as &$u) {
                $lastAt = (string) ($u['last_message_at'] ?? '');
                $lastVoteAt = isset($u['last_vote_at']) ? (string) $u['last_vote_at'] : '';
                if ($lastVoteAt !== '' && ($lastAt === '' || $lastVoteAt > $lastAt)) {
                    $lastAt = $lastVoteAt;
                }
                $days = null;
                if ($lastAt !== '') {
                    try {
                        $dt = new \DateTimeImmutable($lastAt);
                        $diff = $now->diff($dt);
                        $days = (int) $diff->days;
                    } catch (\Throwable $e) {
                        $days = null;
                    }
                }

                $status = 'inactive';
                $statusLabel = 'غیرفعال';
                $votesCast = isset($u['votes_cast']) ? (int) $u['votes_cast'] : 0;
                $total = (int) $u['total_messages'] + $votesCast;

                if ($days !== null && $days <= 7 && $total >= 20) {
                    $status = 'active';
                    $statusLabel = 'فعال 🔥';
                } elseif ($days !== null && $days <= 30 && $total >= 5) {
                    $status = 'semi';
                    $statusLabel = 'نیمه‌فعال';
                }

                $u['status'] = $status;
                $u['status_label'] = $statusLabel;

                $messagesTrading = (int) ($u['messages_trading'] ?? 0);
                $messagesWar = (int) ($u['messages_war'] ?? 0);
                $messagesQa = (int) ($u['messages_qa'] ?? 0);
                $ratedAnalyses = (int) ($u['rated_analyses'] ?? 0);
                $ratingsSum = (int) ($u['ratings_sum'] ?? 0);
                $votesCast = (int) ($u['votes_cast'] ?? 0);

                $scoreTradingRaw = ($messagesTrading * 1.0)
                    + ($ratedAnalyses * 3.0)
                    + ($ratingsSum * 2.0);
                // هر ۱۰ پیام در اقتصاد/جنگ و پرسش‌وپاسخ = ۱ امتیاز
                $scoreWarRaw = (float) floor($messagesWar / 10);
                $scoreQaRaw = (float) floor($messagesQa / 10);
                $scoreVotesRaw = $votesCast * 0.5;

                $totalScore = (int) round($scoreTradingRaw + $scoreWarRaw + $scoreQaRaw + $scoreVotesRaw);

                if ($totalScore < 0) {
                    $totalScore = 0;
                }

                $u['total_score'] = $totalScore;

                if ($totalScore >= 900) {
                    $u['level_label'] = 'سطح ۵ - اسطوره مارکت 👑';
                } elseif ($totalScore >= 400) {
                    $u['level_label'] = 'سطح ۴ - استراتژیست حرفه‌ای 🧠';
                } elseif ($totalScore >= 250) {
                    $u['level_label'] = 'سطح ۳ - تریدر پایدار 🔥';
                } elseif ($totalScore >= 200) {
                    $u['level_label'] = 'سطح ۲ - فعال رو به رشد 🚀';
                } else {
                    $u['level_label'] = 'سطح ۱ - تازه‌کار بازار 🐣';
                }
            }
            unset($u);

            $list = array_values($perUser);
            usort($list, function (array $a, array $b): int {
                $sa = (int) ($a['total_score'] ?? 0);
                $sb = (int) ($b['total_score'] ?? 0);
                if ($sa === $sb) {
                    return $b['total_messages'] <=> $a['total_messages'];
                }
                return $sb <=> $sa;
            });

            $linesActive = [];
            $linesSemi = [];
            $linesInactive = [];

            foreach ($list as $u) {
                $name = (string) ($u['first_name'] ?? '');
                if ($name === '' && $u['username'] !== '') {
                    $name = '@' . $u['username'];
                }
                if ($name === '') {
                    $name = 'کاربر ' . $u['id'];
                }

                $votesCast = isset($u['votes_cast']) ? (int) $u['votes_cast'] : 0;
                $totalMessages = (int) ($u['total_messages'] ?? 0);
                $messagesTrading = (int) ($u['messages_trading'] ?? 0);
                $messagesWar = (int) ($u['messages_war'] ?? 0);
                $messagesQa = (int) ($u['messages_qa'] ?? 0);
                $ratedAnalyses = (int) ($u['rated_analyses'] ?? 0);
                $ratingsCount = (int) ($u['ratings_count'] ?? 0);
                $totalScore = (int) ($u['total_score'] ?? 0);

                // خط اول: هویت و وضعیت کلی هنرجو
                $line = '👤 <b>' . $name . '</b> | 🆔 <code>' . $u['id'] . '</code> | ' . $u['level_label'] . ' | ' . $u['status_label'] . "\n";
                // خط دوم: آمار فشرده پیام‌ها و امتیازها
                $line .= '✉️ ' . $totalMessages
                    . ' | 💹 ' . $messagesTrading
                    . ' | ⚔️ ' . $messagesWar
                    . ' | ❓ ' . $messagesQa
                    . ' | ⭐ ' . $totalScore
                    . ' | 📊 ' . $ratedAnalyses
                    . ' | 🤝 ' . $votesCast;

                if ($u['status'] === 'active') {
                    $linesActive[] = $line;
                } elseif ($u['status'] === 'semi') {
                    $linesSemi[] = $line;
                } else {
                    $linesInactive[] = $line;
                }
            }

            $parts = [];
            if (!empty($linesActive)) {
                $parts[] = "🔥 فعال‌ها:\n\n" . implode("\n\n", $linesActive);
            }
            if (!empty($linesSemi)) {
                $parts[] = "🟡 نیمه‌فعال‌ها:\n\n" . implode("\n\n", $linesSemi);
            }
            if (!empty($linesInactive)) {
                $parts[] = "⚫ غیرفعال‌ها:\n\n" . implode("\n\n", $linesInactive);
            }

            $text = "<b>👥 وضعیت هنرجویان در گروه اصلی</b>\n\n" . implode("\n\n\n", $parts);

            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '⬅️ بازگشت به منوی ادمین', 'callback_data' => 'admin:menu'],
                    ],
                ],
            ];

            if ($messageId > 0) {
                $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            } else {
                $bot->getClient()->sendMessage($chatId, $text, [
                    'reply_markup' => $keyboard,
                ]);
            }
        }
    }

    private function handleInstallGroup(array $update, TelegramBot $bot): void
    {
        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'], $chat['type'])) {
            return;
        }

        $chatType = (string) $chat['type'];
        if ($chatType !== 'group' && $chatType !== 'supergroup') {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];
        $storage = $bot->getStorage();
        $settings = $storage->load('settings');

        if (isset($settings['main_group']['id']) && (int) $settings['main_group']['id'] === $chatId) {
            $bot->getClient()->sendMessage($chatId, 'این گروه قبلاً به عنوان گروه اصلی ربات ثبت شده است.');
            return;
        }

        $storage->update('settings', function (array $s) use ($chat, $chatId): array {
            $now = (new \DateTimeImmutable())->format('Y-m-d H:i:s');

            $s['main_group'] = [
                'id' => $chatId,
                'title' => (string) ($chat['title'] ?? ''),
                'username' => (string) ($chat['username'] ?? ''),
                'type' => (string) ($chat['type'] ?? ''),
                'set_at' => $now,
            ];

            return $s;
        });

        if (isset($settings['main_group']['id']) && (int) $settings['main_group']['id'] !== $chatId) {
            $prevId = (int) $settings['main_group']['id'];
            $bot->getClient()->sendMessage($chatId, 'گروه اصلی ربات از گروه قبلی (ID: ' . $prevId . ') به این گروه تغییر کرد.');
        } else {
            $text = "این گروه به عنوان گروه اصلی ربات ثبت شد.\n";
            $text .= "از این به بعد داشبورد و آمار، بر اساس این گروه نمایش داده می‌شود.";
            $bot->getClient()->sendMessage($chatId, $text);

            $warnText = "⚠️ <b>شروع سیستم پایش فعالیت</b>\n\n";
            $warnText .= "اعضای محترم گروه ترید 👥\n";
            $warnText .= "از این لحظه، <b>پیام‌ها، فعالیت در تاپیک «ترید هنرجویان» و امتیازدهی به تحلیل‌ها</b> به‌صورت منظم ثبت می‌شود تا کوچ بتواند <b>روند پیشرفت شما</b> را دقیق‌تر دنبال کند.\n\n";
            $warnText .= "🔔 در صورت دوره‌های طولانی عدم‌فعالیت، تنها <b>یادآوری‌های دوستانه</b> برای شما و <b>گزارش</b> برای کوچ ارسال می‌شود. هیچ‌گونه <b>بن یا اخراج خودکار</b> توسط ربات انجام نخواهد شد و هر تصمیم صرفاً با نظر کوچ و مدیریت گروه است.";

            $bot->getClient()->sendMessage($chatId, $warnText);
        }
    }

    private function handleSetTopicRole(array $update, TelegramBot $bot): void
    {
        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'], $chat['type'])) {
            return;
        }

        $chatType = (string) $chat['type'];
        if ($chatType !== 'group' && $chatType !== 'supergroup') {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $text = isset($message['text']) ? trim((string) $message['text']) : '';
        $parts = explode(' ', $text, 2);
        if (count($parts) < 2) {
            return;
        }
        $role = strtolower(trim($parts[1]));

        $allowedRoles = ['war', 'qa', 'trading'];
        if (!in_array($role, $allowedRoles, true)) {
            return;
        }

        if ($role !== 'qa' && !isset($message['message_thread_id'])) {
            return;
        }

        $chatId = (int) $chat['id'];
        $topicId = isset($message['message_thread_id']) ? (int) $message['message_thread_id'] : 0;

        $bot->getStorage()->update('topic_roles', function (array $roles) use ($chatId, $topicId, $role): array {
            if (!isset($roles[$chatId])) {
                $roles[$chatId] = [];
            }

            $roles[$chatId][$role] = [
                'topic_id' => $topicId,
            ];

            return $roles;
        });

        $labels = [
            'war' => 'اقتصاد و جنگ',
            'qa' => 'پرسش و پاسخ',
            'trading' => 'ترید هنرجویان',
        ];

        $label = $labels[$role] ?? $role;

        $reply = 'این تاپیک به عنوان تاپیک «' . $label . '» ثبت شد.';

        $bot->getClient()->sendMessage($chatId, $reply, [
            'reply_to_message_id' => (int) ($message['message_id'] ?? 0),
        ]);
    }

    private function handleStudentLookupCommand(array $update, TelegramBot $bot): void
    {
        $message = $update['message'] ?? [];
        $from = $message['from'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'])) {
            return;
        }

        $fromId = (int) $from['id'];
        $chatId = isset($chat['id']) ? (int) $chat['id'] : $fromId;

        $storage = $bot->getStorage();
        $config = $bot->getConfig();

        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $text = isset($message['text']) ? trim((string) $message['text']) : '';
        $parts = explode(' ', $text, 2);
        $arg = isset($parts[1]) ? trim($parts[1]) : '';

        $adminIds = $config['admin_ids'] ?? [];
        $isAdmin = in_array($fromId, array_map('intval', $adminIds), true);

        $targetId = $fromId;

        $users = $storage->load('users');
        if (!is_array($users)) {
            $users = [];
        }

        if ($arg !== '' && $isAdmin) {
            $candidateId = null;

            if ($arg[0] === '@') {
                $username = substr($arg, 1);
                $usernameLower = strtolower($username);
                foreach ($users as $u) {
                    if (!is_array($u)) {
                        continue;
                    }
                    $uUsername = strtolower((string) ($u['username'] ?? ''));
                    if ($uUsername !== '' && $uUsername === $usernameLower) {
                        $candidateId = (int) $u['id'];
                        break;
                    }
                }
            } elseif (ctype_digit($arg)) {
                $candidateId = (int) $arg;
            } else {
                $needle = strtolower($arg);
                foreach ($users as $u) {
                    if (!is_array($u)) {
                        continue;
                    }
                    $first = strtolower((string) ($u['first_name'] ?? ''));
                    $user = strtolower((string) ($u['username'] ?? ''));
                    if ($first === $needle || $user === $needle) {
                        $candidateId = (int) $u['id'];
                        break;
                    }
                }
            }

            if ($candidateId === null) {
                $bot->getClient()->sendMessage($chatId, 'هیچ هنرجویی با این مشخصات پیدا نشد. از شناسه عددی یا @username استفاده کن.');
                return;
            }

            $targetId = $candidateId;
        }

        $topicActivity = $storage->load('topic_activity');
        $topicRoles = $storage->load('topic_roles');
        $ratings = $storage->load('ratings');

        $rolesForChat = isset($topicRoles[$mainGroupId]) && is_array($topicRoles[$mainGroupId]) ? $topicRoles[$mainGroupId] : [];
        $tradingTopicId = isset($rolesForChat['trading']['topic_id']) ? (int) $rolesForChat['trading']['topic_id'] : null;
        $warTopicId = isset($rolesForChat['war']['topic_id']) ? (int) $rolesForChat['war']['topic_id'] : null;
        $qaTopicId = isset($rolesForChat['qa']['topic_id']) ? (int) $rolesForChat['qa']['topic_id'] : null;

        $stats = [
            'total_messages' => 0,
            'messages_trading' => 0,
            'messages_war' => 0,
            'messages_qa' => 0,
            'messages_other' => 0,
            'first_message_at' => '',
            'last_message_at' => '',
            'rated_analyses' => 0,
            'ratings_sum' => 0,
            'ratings_count' => 0,
            'votes_cast' => 0,
            'last_vote_at' => '',
            'media_total' => 0,
            'links_total' => 0,
        ];

        if (isset($topicActivity[$mainGroupId]) && is_array($topicActivity[$mainGroupId])) {
            foreach ($topicActivity[$mainGroupId] as $topicId => $topic) {
                if (!is_array($topic) || !isset($topic['users'][$targetId])) {
                    continue;
                }

                $role = 'other';
                if ($tradingTopicId !== null && (int) $topicId === $tradingTopicId) {
                    $role = 'trading';
                } elseif ($warTopicId !== null && (int) $topicId === $warTopicId) {
                    $role = 'war';
                } elseif ($qaTopicId !== null && (int) $topicId === $qaTopicId) {
                    $role = 'qa';
                }

                $u = $topic['users'][$targetId];
                $count = (int) ($u['message_count'] ?? 0);
                $stats['total_messages'] += $count;

                $mediaCount = (int) ($u['media_count'] ?? 0);
                $linkCount = (int) ($u['link_count'] ?? 0);
                $stats['media_total'] += $mediaCount;
                $stats['links_total'] += $linkCount;

                if ($role === 'trading') {
                    $stats['messages_trading'] += $count;
                } elseif ($role === 'war') {
                    $stats['messages_war'] += $count;
                } elseif ($role === 'qa') {
                    $stats['messages_qa'] += $count;
                } else {
                    $stats['messages_other'] += $count;
                }

                $uFirst = (string) ($u['first_message_at'] ?? '');
                $uLast = (string) ($u['last_message_at'] ?? '');

                if ($stats['first_message_at'] === '' || ($uFirst !== '' && $uFirst < $stats['first_message_at'])) {
                    $stats['first_message_at'] = $uFirst;
                }
                if ($uLast !== '' && ($stats['last_message_at'] === '' || $uLast > $stats['last_message_at'])) {
                    $stats['last_message_at'] = $uLast;
                }
            }
        }

        if (isset($ratings[$mainGroupId]) && is_array($ratings[$mainGroupId])) {
            foreach ($ratings[$mainGroupId] as $entry) {
                if (!is_array($entry)) {
                    continue;
                }

                if ((int) ($entry['author_id'] ?? 0) === $targetId) {
                    $stats['rated_analyses']++;

                    $entryStats = $entry['stats'] ?? [];
                    $sum = (int) ($entryStats['sum'] ?? 0);
                    $count = (int) ($entryStats['count'] ?? 0);

                    $stats['ratings_sum'] += $sum;
                    $stats['ratings_count'] += $count;
                }

                if (isset($entry['votes']) && is_array($entry['votes']) && isset($entry['votes'][$targetId])) {
                    $vote = $entry['votes'][$targetId];
                    $stats['votes_cast']++;

                    $votedAt = (string) ($vote['voted_at'] ?? '');
                    if ($votedAt !== '') {
                        if ($stats['last_vote_at'] === '' || $votedAt > $stats['last_vote_at']) {
                            $stats['last_vote_at'] = $votedAt;
                        }
                    }
                }
            }
        }

        $now = new \DateTimeImmutable('now');
        $status = 'inactive';
        $statusLabel = 'غیرفعال';

        $days = null;
        $lastAt = $stats['last_message_at'];
        if ($stats['last_vote_at'] !== '' && ($lastAt === '' || $stats['last_vote_at'] > $lastAt)) {
            $lastAt = $stats['last_vote_at'];
        }

        if ($lastAt !== '') {
            try {
                $dt = new \DateTimeImmutable($lastAt);
                $diff = $now->diff($dt);
                $days = (int) $diff->days;
            } catch (\Throwable $e) {
                $days = null;
            }
        }

        $totalMessages = (int) $stats['total_messages'] + (int) $stats['votes_cast'];
        if ($days !== null && $days <= 7 && $totalMessages >= 20) {
            $status = 'active';
            $statusLabel = 'فعال 🔥';
        } elseif ($days !== null && $days <= 30 && $totalMessages >= 5) {
            $status = 'semi';
            $statusLabel = 'نیمه‌فعال';
        }

        // امتیاز فعالیت به تفکیک تاپیک و در مجموع
        $messagesTrading = (int) $stats['messages_trading'];
        $messagesWar = (int) $stats['messages_war'];
        $messagesQa = (int) $stats['messages_qa'];
        $ratedAnalyses = (int) $stats['rated_analyses'];
        $ratingsSum = (int) $stats['ratings_sum'];
        $votesCast = (int) $stats['votes_cast'];

        $scoreTradingRaw = ($messagesTrading * 1.0)
            + ($ratedAnalyses * 3.0)
            + ($ratingsSum * 2.0);
        // هر ۱۰ پیام در اقتصاد/جنگ و پرسش‌وپاسخ = ۱ امتیاز
        $scoreWarRaw = (float) floor($messagesWar / 10);
        $scoreQaRaw = (float) floor($messagesQa / 10);
        $scoreVotesRaw = $votesCast * 0.5;

        $scoreTrading = (int) round($scoreTradingRaw);
        $scoreWar = (int) $scoreWarRaw;
        $scoreQa = (int) $scoreQaRaw;
        $scoreVotes = (int) round($scoreVotesRaw);
        $totalScore = (int) round($scoreTradingRaw + $scoreWarRaw + $scoreQaRaw + $scoreVotesRaw);

        if ($totalScore < 0) {
            $totalScore = 0;
        }

        if ($totalScore >= 900) {
            $levelLabel = 'سطح ۵ - اسطوره مارکت 👑';
        } elseif ($totalScore >= 400) {
            $levelLabel = 'سطح ۴ - استراتژیست حرفه‌ای 🧠';
        } elseif ($totalScore >= 250) {
            $levelLabel = 'سطح ۳ - تریدر پایدار 🔥';
        } elseif ($totalScore >= 200) {
            $levelLabel = 'سطح ۲ - فعال رو به رشد 🚀';
        } else {
            $levelLabel = 'سطح ۱ - تازه‌کار بازار 🐣';
        }

        $userData = $users[$targetId] ?? [];

        $name = '';
        if (isset($userData['first_name']) && $userData['first_name'] !== '') {
            $name = (string) $userData['first_name'];
        } elseif (isset($userData['username']) && $userData['username'] !== '') {
            $name = '@' . (string) $userData['username'];
        } else {
            $name = 'کاربر ' . $targetId;
        }

        $lines = [];
        $lines[] = '👤 آمار عملکرد ' . $name;
        $lines[] = 'شناسه: <code>' . $targetId . '</code>';
        $lines[] = 'سطح فعلی: ' . $levelLabel . ' (امتیاز کل فعالیت: ' . $totalScore . ')';
        $lines[] = 'وضعیت فعالیت: ' . $statusLabel;

        $lines[] = '';
        $lines[] = '<b>🏅 امتیاز فعالیت:</b>';
        $lines[] = '🔥 تریدینگ (پیام + تحلیل): ' . $scoreTrading;
        $lines[] = '⚔️ اقتصاد و جنگ: ' . $messagesWar . ' پیام (' . $scoreWar . ' امتیاز)';
        $lines[] = '❓ پرسش و پاسخ: ' . $messagesQa . ' پیام (' . $scoreQa . ' امتیاز)';
        $lines[] = '🤝 مشارکت در رأی‌دهی: ' . $scoreVotes;
        $lines[] = '🏆 امتیاز کل فعالیت: ' . $totalScore;

        $lines[] = '';
        $lines[] = '<b>📊 پیام‌ها در گروه اصلی:</b>';
        $lines[] = 'کل پیام‌ها: ' . $stats['total_messages'];
        $lines[] = '• تاپیک ترید: ' . $stats['messages_trading'];
        $lines[] = '• اقتصاد و جنگ: ' . $stats['messages_war'];
        $lines[] = '• پرسش و پاسخ: ' . $stats['messages_qa'];

        if ($stats['messages_other'] > 0) {
            $lines[] = '• سایر تاپیک‌ها: ' . $stats['messages_other'];
        }

        $lines[] = '';
        $lines[] = '<b>📎 مدیا و لینک‌ها:</b>';
        $lines[] = 'مدیا (عکس/ویدیو/فایل): ' . $stats['media_total'];
        $lines[] = 'لینک‌ها: ' . $stats['links_total'];

        if ($stats['first_message_at'] !== '' && $stats['last_message_at'] !== '') {
            $lines[] = '';
            $lines[] = 'اولین پیام ثبت‌شده: ' . $this->formatJalali($stats['first_message_at']);
            $lines[] = 'آخرین فعالیت ثبت‌شده: ' . $this->formatJalali($lastAt);
        }

        $lines[] = '';
        $lines[] = '<b>⭐ امتیازدهی تحلیل‌ها:</b>';
        $lines[] = 'تعداد تحلیل‌های دارای امتیاز: ' . $stats['rated_analyses'];
        $lines[] = 'مجموع ستاره‌های دریافت‌شده: ' . $stats['ratings_sum'];
        $lines[] = 'تعداد رأی‌ها (نفر-رأی): ' . $stats['ratings_count'];

        if ($stats['votes_cast'] > 0) {
            $lines[] = '🤝 تعداد رأی‌هایی که برای تحلیل دیگران ثبت کرده‌ای: ' . $stats['votes_cast'];
        }

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔍 جزئیات فعالیت (۱۴ روز اخیر)', 'callback_data' => 'student:details:' . $targetId],
                ],
            ],
        ];

        $bot->getClient()->sendMessage($chatId, implode("\n", $lines), [
            'reply_markup' => $keyboard,
        ]);
    }

    private function handleStudentStatsCallback(string $data, array $update, TelegramBot $bot): void
    {
        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $chatId = (int) $chat['id'];
        $messageId = isset($message['message_id']) ? (int) $message['message_id'] : 0;
        $viewerId = (int) $from['id'];

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            $adminIds = [];
        }
        $adminIds = array_map('intval', $adminIds);

        $parts = explode(':', $data);
        if (count($parts) < 3) {
            return;
        }

        $action = $parts[1];
        $targetId = (int) $parts[2];

        $isAdmin = in_array($viewerId, $adminIds, true);

        if ($viewerId !== $targetId && !$isAdmin) {
            $bot->getClient()->sendMessage($chatId, 'این دکمه فقط برای خود هنرجو و ادمین قابل استفاده است.');
            return;
        }

        if ($messageId <= 0) {
            return;
        }

        if ($action === 'details') {
            $bot->getClient()->deleteMessage($chatId, $messageId);
            $this->sendStudentDetailsView($chatId, $targetId, $bot);
        } elseif ($action === 'back') {
            $bot->getClient()->deleteMessage($chatId, $messageId);
            $fakeUpdate = [
                'message' => [
                    'from' => ['id' => $viewerId],
                    'chat' => ['id' => $chatId],
                    'text' => $isAdmin ? '/student ' . $targetId : '/student',
                ],
            ];
            $this->handleStudentLookupCommand($fakeUpdate, $bot);
        }
    }

    private function sendStudentDetailsView(int $chatId, int $targetId, TelegramBot $bot): void
    {
        $storage = $bot->getStorage();
        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $topicActivity = $storage->load('topic_activity');
        $topicRoles = $storage->load('topic_roles');
        $ratings = $storage->load('ratings');
        $users = $storage->load('users');

        if (!is_array($topicActivity)) {
            $topicActivity = [];
        }
        if (!is_array($topicRoles)) {
            $topicRoles = [];
        }
        if (!is_array($ratings)) {
            $ratings = [];
        }
        if (!is_array($users)) {
            $users = [];
        }

        $rolesForChat = isset($topicRoles[$mainGroupId]) && is_array($topicRoles[$mainGroupId]) ? $topicRoles[$mainGroupId] : [];
        $tradingTopicId = isset($rolesForChat['trading']['topic_id']) ? (int) $rolesForChat['trading']['topic_id'] : null;
        $warTopicId = isset($rolesForChat['war']['topic_id']) ? (int) $rolesForChat['war']['topic_id'] : null;
        $qaTopicId = isset($rolesForChat['qa']['topic_id']) ? (int) $rolesForChat['qa']['topic_id'] : null;

        $now = new \DateTimeImmutable('now');
        $windowStart = $now->modify('-13 days')->format('Y-m-d');

        $messages14Total = 0;
        $messages14Trading = 0;
        $messages14War = 0;
        $messages14Qa = 0;
        $perDayTotals = [];

        if (isset($topicActivity[$mainGroupId]) && is_array($topicActivity[$mainGroupId])) {
            foreach ($topicActivity[$mainGroupId] as $topicId => $topic) {
                if (!is_array($topic) || !isset($topic['users']) || !is_array($topic['users'])) {
                    continue;
                }

                if (!isset($topic['users'][$targetId])) {
                    continue;
                }

                $role = 'other';
                if ($tradingTopicId !== null && (int) $topicId === $tradingTopicId) {
                    $role = 'trading';
                } elseif ($warTopicId !== null && (int) $topicId === $warTopicId) {
                    $role = 'war';
                } elseif ($qaTopicId !== null && (int) $topicId === $qaTopicId) {
                    $role = 'qa';
                }

                $u = $topic['users'][$targetId];
                $daily = isset($u['daily_messages']) && is_array($u['daily_messages']) ? $u['daily_messages'] : [];
                foreach ($daily as $day => $count) {
                    if (!is_string($day) || $day === '') {
                        continue;
                    }
                    if ($day < $windowStart) {
                        continue;
                    }
                    $count = (int) $count;
                    if ($count <= 0) {
                        continue;
                    }
                    $messages14Total += $count;
                    if ($role === 'trading') {
                        $messages14Trading += $count;
                    } elseif ($role === 'war') {
                        $messages14War += $count;
                    } elseif ($role === 'qa') {
                        $messages14Qa += $count;
                    }

                    if (!isset($perDayTotals[$day])) {
                        $perDayTotals[$day] = [
                            'total' => 0,
                            'votes' => 0,
                        ];
                    }
                    $perDayTotals[$day]['total'] += $count;
                }
            }
        }

        $votes14 = 0;
        if (isset($ratings[$mainGroupId]) && is_array($ratings[$mainGroupId])) {
            foreach ($ratings[$mainGroupId] as $entry) {
                if (!is_array($entry)) {
                    continue;
                }
                if (!isset($entry['votes']) || !is_array($entry['votes'])) {
                    continue;
                }
                foreach ($entry['votes'] as $voterId => $vote) {
                    $voterId = (int) $voterId;
                    if ($voterId !== $targetId) {
                        continue;
                    }
                    $votedAt = (string) ($vote['voted_at'] ?? '');
                    if ($votedAt === '') {
                        continue;
                    }
                    try {
                        $dt = new \DateTimeImmutable($votedAt);
                    } catch (\Throwable $e) {
                        continue;
                    }
                    $day = $dt->format('Y-m-d');
                    if ($day < $windowStart) {
                        continue;
                    }
                    $votes14++;
                    if (!isset($perDayTotals[$day])) {
                        $perDayTotals[$day] = [
                            'total' => 0,
                            'votes' => 0,
                        ];
                    }
                    $perDayTotals[$day]['votes']++;
                }
            }
        }

        // محاسبه تعداد روزهای فعال در ۱۴ روز اخیر
        $activeDays14 = 0;
        foreach ($perDayTotals as $info) {
            $total = (int) ($info['total'] ?? 0);
            $votes = (int) ($info['votes'] ?? 0);
            if ($total > 0 || $votes > 0) {
                $activeDays14++;
            }
        }

        // الگوی روزهای هفته (براساس مجموع پیام + رأی در ۱۴ روز)
        $weekdayTotals = [
            1 => 0,
            2 => 0,
            3 => 0,
            4 => 0,
            5 => 0,
            6 => 0,
            7 => 0,
        ];

        foreach ($perDayTotals as $day => $info) {
            $total = (int) ($info['total'] ?? 0) + (int) ($info['votes'] ?? 0);
            if ($total <= 0 || !is_string($day) || $day === '') {
                continue;
            }
            try {
                $dt = new \DateTimeImmutable($day . ' 00:00:00');
            } catch (\Throwable $e) {
                continue;
            }
            $w = (int) $dt->format('N');
            if (!isset($weekdayTotals[$w])) {
                $weekdayTotals[$w] = 0;
            }
            $weekdayTotals[$w] += $total;
        }

        $weekdayNames = [
            6 => 'شنبه',
            7 => 'یکشنبه',
            1 => 'دوشنبه',
            2 => 'سه‌شنبه',
            3 => 'چهارشنبه',
            4 => 'پنجشنبه',
            5 => 'جمعه',
        ];

        $weekdayLines = [];
        foreach ([6, 7, 1, 2, 3, 4, 5] as $w) {
            $val = (int) ($weekdayTotals[$w] ?? 0);
            if ($val <= 0) {
                continue;
            }
            $nameW = $weekdayNames[$w] ?? (string) $w;
            $weekdayLines[] = '• ' . $nameW . ': ' . $val . ' فعالیت';
        }

        $peakLine = '';
        $maxWeek = max($weekdayTotals);
        if ($maxWeek > 0) {
            $peakDays = [];
            foreach ($weekdayTotals as $w => $val) {
                if ($val === $maxWeek && isset($weekdayNames[$w])) {
                    $peakDays[] = $weekdayNames[$w];
                }
            }
            if (!empty($peakDays)) {
                $peakLine = 'روزهای اوج فعالیت: ' . implode('، ', $peakDays);
            }
        }

        $last7Lines = [];
        for ($i = 6; $i >= 0; $i--) {
            $dayDate = $now->modify('-' . $i . ' days');
            $dayKey = $dayDate->format('Y-m-d');
            $counts = $perDayTotals[$dayKey] ?? ['total' => 0, 'votes' => 0];
            $total = (int) ($counts['total'] ?? 0);
            $votes = (int) ($counts['votes'] ?? 0);
            $label = $this->formatJalali($dayKey . ' 00:00:00');
            $prefix = ($total > 0 || $votes > 0) ? '✅ ' : '⚪ ';
            $last7Lines[] = $prefix . $label . ' — ✉️ ' . $total . ' | 🤝 ' . $votes;
        }

        $userData = $users[$targetId] ?? [];
        $name = '';
        if (is_array($userData) && isset($userData['first_name']) && $userData['first_name'] !== '') {
            $name = (string) $userData['first_name'];
        } elseif (is_array($userData) && isset($userData['username']) && $userData['username'] !== '') {
            $name = '@' . (string) $userData['username'];
        } else {
            $name = 'کاربر ' . $targetId;
        }

        $lines = [];
        $lines[] = '';
        $lines[] = '<b>📊 الگوی روزهای هفته:</b>';
        foreach ($weekdayLines as $wl) {
            $lines[] = $wl;
        }
        if ($peakLine !== '') {
            $lines[] = $peakLine;
        }
        $lines[] = '🔍 جزئیات فعالیت ' . $name . ' (۱۴ روز اخیر)';
        $lines[] = '';
        $lines[] = '📅 روزهای فعال: ' . $activeDays14 . ' از ۱۴ روز';
        $lines[] = '✉️ مجموع پیام‌ها (۱۴ روز): ' . $messages14Total;
        $lines[] = '💹 پیام‌های تریدینگ (۱۴ روز): ' . $messages14Trading;
        $lines[] = '⚔️ پیام‌های اقتصاد/جنگ (۱۴ روز): ' . $messages14War;
        $lines[] = '❓ پیام‌های پرسش و پاسخ (۱۴ روز): ' . $messages14Qa;
        $lines[] = '🤝 رأی‌ها (۱۴ روز): ' . $votes14;
        $lines[] = '';
        $lines[] = '<b>📆 الگوی ۷ روز اخیر:</b>';
        foreach ($last7Lines as $l) {
            $lines[] = $l;
        }

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬅️ بازگشت به آمار اصلی', 'callback_data' => 'student:back:' . $targetId],
                ],
            ],
        ];

        $bot->getClient()->sendMessage($chatId, implode("\n", $lines), [
            'reply_markup' => $keyboard,
        ]);
    }

    private function formatJalali(string $datetime): string
    {
        if ($datetime === '') {
            return '';
        }

        try {
            $dt = new \DateTimeImmutable($datetime);
        } catch (\Throwable $e) {
            return $datetime;
        }

        if (class_exists('IntlDateFormatter')) {
            $formatter = new \IntlDateFormatter(
                'fa_IR@calendar=persian',
                \IntlDateFormatter::SHORT,
                \IntlDateFormatter::SHORT,
                $dt->getTimezone()->getName(),
                \IntlDateFormatter::TRADITIONAL,
                'yyyy/MM/dd HH:mm'
            );

            $formatted = $formatter->format($dt);
            if ($formatted !== false) {
                $en = ['0','1','2','3','4','5','6','7','8','9'];
                $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
                return str_replace($en, $fa, $formatted);
            }
        }

        // Fallback: تقریب شمسی به صورت دستی بر اساس تاریخ میلادی
        $gy = (int) $dt->format('Y');
        $gm = (int) $dt->format('n');
        $gd = (int) $dt->format('j');

        [$jy, $jm, $jd] = $this->gregorianToJalali($gy, $gm, $gd);

        $time = $dt->format('H:i');
        $formatted = sprintf('%04d/%02d/%02d %s', $jy, $jm, $jd, $time);

        $en = ['0','1','2','3','4','5','6','7','8','9'];
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹'];
        return str_replace($en, $fa, $formatted);
    }

    /**
     * تبدیل تاریخ میلادی به جلالی (تقویم شمسی)
     * الگوریتم استاندارد بر اساس شمارش روز مطلق.
     */
    private function gregorianToJalali(int $gy, int $gm, int $gd): array
    {
        $gMonthDay = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334];

        $gy2 = $gy - 1600;
        $gm2 = $gm - 1;
        $gd2 = $gd - 1;

        $gDayNo = 365 * $gy2
            + (int) (($gy2 + 3) / 4)
            - (int) (($gy2 + 99) / 100)
            + (int) (($gy2 + 399) / 400);

        $gDayNo += $gMonthDay[$gm2] + $gd2;

        // سال کبیسه میلادی بعد از فوریه
        if ($gm2 > 1 && (($gy % 4 === 0 && $gy % 100 !== 0) || ($gy % 400 === 0))) {
            $gDayNo++;
        }

        $jDayNo = $gDayNo - 79;

        $jNp = (int) ($jDayNo / 12053); // 12053 = 365*33 + 8
        $jDayNo %= 12053;

        $jy = 979 + 33 * $jNp + 4 * (int) ($jDayNo / 1461);
        $jDayNo %= 1461;

        if ($jDayNo >= 366) {
            $jy += (int) (($jDayNo - 366) / 365);
            $jDayNo = ($jDayNo - 366) % 365;
        }

        $jm = 0;
        for ($i = 0; $i < 11 && $jDayNo >= 0; $i++) {
            $daysInMonth = ($i < 6) ? 31 : 30;
            if ($jDayNo < $daysInMonth) {
                $jm = $i + 1;
                break;
            }
            $jDayNo -= $daysInMonth;
        }

        $jd = $jDayNo + 1;

        return [$jy, $jm, $jd];
    }
}
